import ccxt
import json
import time
import os
import requests
import pandas as pd
import csv
from ta.momentum import RSIIndicator
from ta.trend import EMAIndicator
from dotenv import load_dotenv
from datetime import datetime

load_dotenv()

# Configuración
API_KEY = os.getenv("BINANCE_API_KEY")
API_SECRET = os.getenv("BINANCE_API_SECRET")
TELEGRAM_TOKEN = os.getenv("TELEGRAM_BOT_TOKEN")
CHAT_ID = os.getenv("TELEGRAM_CHAT_ID")
SYMBOLS = ['PEPE/USDT', 'BTC/USDC']
STOP_LOSS_PERCENT_MEME = 50
STOP_LOSS_PERCENT_BTC = 50
STOP_LOSS_PERCENT = 75
PROFIT_PERCENT_MEME = 0.55
PROFIT_PERCENT_BTC = 0.55
TIMEFRAME = '1m'
TOPE_COMPRA_MEME = 0.99
TOPE_COMPRA_BTC = 0.99
RSI_LIMIT_BTC = 60
RSI_LIMIT_MEME = 60

binance = ccxt.binance({
    'apiKey': API_KEY,
    'secret': API_SECRET,
    'enableRateLimit': True,
    'options': {'defaultType': 'spot'}
})

BASE_DIR = os.path.dirname(os.path.abspath(__file__))
LOG_FILE = os.path.join(BASE_DIR, "trades_log.csv")


# ---------- FUNCIONES AUXILIARES ----------
def get_state_file(symbol):
    return os.path.join(BASE_DIR, f'state_{symbol.replace("/", "_").lower()}.json')


def load_state(symbol):
    file = get_state_file(symbol)
    if os.path.exists(file):
        with open(file) as f:
            return json.load(f)
    return {"position": False, "buy_price": 0.0, "sell_order_id": None}


def save_state(symbol, state):
    with open(get_state_file(symbol), 'w') as f:
        json.dump(state, f)


def send_telegram(msg):
    url = f"https://api.telegram.org/bot{TELEGRAM_TOKEN}/sendMessage"
    payload = {"chat_id": CHAT_ID, "text": msg}
    requests.post(url, data=payload)


def fetch_ohlcv(symbol, timeframe=TIMEFRAME, limit=120):
    return binance.fetch_ohlcv(symbol, timeframe, limit=limit)


def calculate_indicators(ohlcv):
    closes = [candle[4] for candle in ohlcv]
    rsi = RSIIndicator(pd.Series(closes)).rsi().iloc[-1]
    ema = EMAIndicator(pd.Series(closes), window=12).ema_indicator().iloc[-1]
    return rsi, ema, closes[-1]


# ---------- NUEVA FUNCIÓN place_order ----------
def place_order(symbol, side, amount, price=None):
    """
    - Compra ('buy'): orden MARKET inmediata.
    - Venta ('sell'): orden LIMIT al precio indicado.
    """
    if side == 'buy':
        return binance.create_market_order(symbol, side, amount)
    elif side == 'sell':
        if price is None:
            raise ValueError("Debe especificar un precio para la orden LIMIT de venta.")
        return binance.create_limit_order(symbol, side, amount, price)


def calculate_ema_series(ohlcv, period):
    closes = [candle[4] for candle in ohlcv]
    df = pd.DataFrame({'close': closes})
    return df['close'].ewm(span=period, adjust=False).mean()


def get_token_amount(symbol):
    token = symbol.split('/')[0]
    balance = binance.fetch_balance()
    return balance['total'].get(token, 0)


def get_base_balance(symbol):
    base = symbol.split('/')[1]
    balance = binance.fetch_balance()
    return balance['total'].get(base, 0)


# ---------- VERIFICAR ESTADO DE ÓRDENES ----------
def check_order_status(symbol, order_id):
    """Devuelve el estado actual de una orden ('open', 'closed', 'canceled')."""
    try:
        order_info = binance.fetch_order(order_id, symbol)
        return order_info
    except Exception as e:
        print(f"Error al verificar estado de orden {order_id}: {str(e)}")
        return None


# ---------- PROCESAR CADA PAR ---------- 
def process_symbol(symbol):
    state = load_state(symbol)
    ohlcv = fetch_ohlcv(symbol)
    rsi, ema12, current_price = calculate_indicators(ohlcv)
    ema50_series = calculate_ema_series(ohlcv, 50)
    ema50_current = ema50_series.iloc[-1]
    ema50_prev = ema50_series.iloc[-2]

    base_balance = get_base_balance(symbol)
    buy_price = state["buy_price"]

    high_prices = [candle[2] for candle in ohlcv[-120:]]
    recent_high = max(high_prices)

    markets = binance.load_markets()
    market = markets[symbol]
    min_amount = market['limits']['amount']['min']

    if "BTC" in symbol:
        profit_percent_used = PROFIT_PERCENT_BTC
        rsi_limit = RSI_LIMIT_BTC
        price_threshold = recent_high * TOPE_COMPRA_BTC
        stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT_BTC / 100)
    else:
        profit_percent_used = PROFIT_PERCENT_MEME
        rsi_limit = RSI_LIMIT_MEME
        price_threshold = recent_high * TOPE_COMPRA_MEME
        stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT_MEME / 100)

    # -------- FILTROS -------- #
    def is_higher_trend_bullish(symbol, timeframe="15m"):
        ohlcv_higher = fetch_ohlcv(symbol, timeframe=timeframe, limit=100)
        closes = [c[4] for c in ohlcv_higher]
        ema50 = pd.Series(closes).ewm(span=50, adjust=False).mean()
        return closes[-1] > ema50.iloc[-1]

    tendencia_alcista = (current_price > ema50_current) and (ema50_current > ema50_prev)

    last_buy_time = state.get("last_buy_time", 0)
    cooldown = 60 * 30
    now = time.time()
    cooldown_ready = (now - last_buy_time) > cooldown

    recent_closes = [c[4] for c in ohlcv[-10:]]
    drop_percent = (recent_closes[0] - recent_closes[-1]) / recent_closes[0] * 100
    not_falling_knife = drop_percent < 3

    # --------🔍 VERIFICAR ORDEN LIMIT ABIERTA -------- #
    sell_order_id = state.get("sell_order_id")
    if sell_order_id:
        order_info = check_order_status(symbol, sell_order_id)
        if order_info:
            status = order_info['status']
            if status == "closed":
                print(f"✅ Orden {sell_order_id} ejecutada para {symbol}")
                state["position"] = False
                state["buy_price"] = 0.0
                state["sell_order_id"] = None
                save_state(symbol, state)
                send_telegram(f"✅ Venta completada automáticamente para {symbol}")
                return

            elif status == "open":
                created_time = order_info['timestamp'] / 1000
                if time.time() - created_time > 259200:  # 72 horas
                    binance.cancel_order(sell_order_id, symbol)
                    state["sell_order_id"] = None
                    save_state(symbol, state)
                    send_telegram(f"🚫 Orden {sell_order_id} cancelada por timeout en {symbol}")
                else:
                    print(f"⏳ Orden {sell_order_id} aún abierta para {symbol}")
                    return

    # --------🟢 COMPRA (MARKET) -------- #
    if not state["position"]:
        if (
            rsi <= rsi_limit
            and current_price >= ema12
            and current_price <= price_threshold
            and tendencia_alcista
            and is_higher_trend_bullish(symbol)
            and cooldown_ready
            and not_falling_knife
        ):
            amount_to_spend = base_balance * 1
            ticker = binance.fetch_ticker(symbol)
            raw_amount = amount_to_spend / ticker['ask']
            amount = float(binance.amount_to_precision(symbol, raw_amount))

            if amount >= min_amount:
                order = place_order(symbol, 'buy', amount)
                state["position"] = True
                state["buy_price"] = order['average']
                state["last_buy_time"] = now
                save_state(symbol, state)

                price = order['average']
                total_usd = price * amount
                fee = order['fees'][0]['cost'] if order.get('fees') else 0.0
                log_trade("BUY", symbol, amount, price, total_usd, fee)
                send_telegram(f"✅ Compra MARKET en {symbol}: {amount:.6f} a {price:.8f}")

    # --------🔴 VENTA (LIMIT) -------- #
    elif state["position"]:
        token_amount = get_token_amount(symbol)
        if token_amount <= 0:
            return

        limit_price = buy_price * (1 + profit_percent_used / 100)
        order = place_order(symbol, 'sell', token_amount, limit_price)
        state["sell_order_id"] = order['id']
        save_state(symbol, state)

        total_usd = limit_price * token_amount
        send_telegram(f"📈 Orden LIMIT colocada para {symbol}\n"
                      f"💰 Precio objetivo: {limit_price:.8f}\n"
                      f"📦 Cantidad: {token_amount:.6f}")


# ---------- REGISTRO CSV ----------
def log_trade(action, symbol, amount, price, total_usd, fee=0.0, profit_loss=0.0):
    file_exists = os.path.isfile(LOG_FILE)
    with open(LOG_FILE, mode="a", newline="") as file:
        writer = csv.writer(file)
        if not file_exists:
            writer.writerow(["Fecha", "Accion", "Symbol", "Cantidad", "Precio", "Total USD", "Comision", "Ganancia/Perdida"])
        writer.writerow([
            datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            action,
            symbol,
            amount,
            price,
            total_usd,
            fee,
            profit_loss
        ])
    
    if "BTC" in symbol:
        profit_target = price * (1 + PROFIT_PERCENT_BTC/ 100)
        # 📢 Notificación por Telegram
        msg = f"📊 {action} {symbol}\n" \
              f"📦 Cantidad {symbol}: {amount:.8f}\n" \
              f"💰 Precio: {price:.8f}\n" \
              f"💵 Total USD: {total_usd:.8f}\n" \
              f"💸 Comisión: {fee:.8f}\n"\
              f"💰 Target Profit: {profit_target:.2f}\n" 
        if action == "SELL":
            total_neto = total_usd - fee
            msg += f"📈 Ganancia/Pérdida: {profit_loss:.8f} USD\n" \
                   f"💰 Total neto: {total_neto:.8f} USD"
        send_telegram(msg)
        
    if "PEPE" in symbol:
        profit_target = price * (1 + PROFIT_PERCENT_MEME/ 100)
        # 📢 Notificación por Telegram
        msg = f"📊 {action} {symbol}\n" \
              f"📦 Cantidad {symbol}: {amount:.0f}\n" \
              f"💰 Precio: {price:.8f}\n" \
              f"💵 Total USD: {total_usd:.8f}\n" \
              f"💸 Comisión: {fee:.8f}\n"\
              f"💰 Target Profit: {profit_target:.8f}\n" 
        if action == "SELL":
            total_neto = total_usd - fee
            msg += f"📈 Ganancia/Pérdida: {profit_loss:.8f} USD\n" \
                   f"💰 Total neto: {total_neto:.8f} USD"
        send_telegram(msg)


# ---------- LOOP PRINCIPAL ----------
def main():
    send_telegram("🤖 Bot multipar iniciado con MARKET BUY + LIMIT SELL")
    last_report_time = 0

    while True:
        try:
            for symbol in SYMBOLS:
                
                process_symbol(symbol)

            # Reporte horario
            current_time = time.time()
            if current_time - last_report_time >= 3600:
                report = "🕒 Reporte horario:\n"
            
                for symbol in SYMBOLS:
                    try:
                        state = load_state(symbol)
                        rsi, ema, current_price = calculate_indicators(fetch_ohlcv(symbol))
                        buy_price = state.get('buy_price', 0.0)
                        in_position = state.get('position', False)
                        stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT / 100)
                        
                        # Máximo de los últimos 7 días (opcional)
                        ohlcv = fetch_ohlcv(symbol, '1m', limit=120)
                        highest_high = max(candle[2] for candle in ohlcv)
                        
                        if "BTC" in symbol:
                            profit_percent_used = PROFIT_PERCENT_BTC  # profit más bajo para scalping BTC
                            threshold_price = highest_high * TOPE_COMPRA_BTC
                            stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT_BTC / 100)
                        else:
                            profit_percent_used = PROFIT_PERCENT_MEME
                            threshold_price = highest_high * TOPE_COMPRA_MEME
                            stop_loss_price = buy_price * (1 - STOP_LOSS_PERCENT_MEME / 100)
                            
                        profit_target = buy_price * (1 + profit_percent_used/ 100)
            
                        if "BTC" in symbol:
                            report += (
                                f"\n📈 {symbol}\n"
                                f"💰 RSI: {rsi:.2f} | EMA12: {ema:.2f} | Precio actual: {current_price:.2f}\n"
                                f"💰 Compra: {buy_price:.2f} | StopLoss: {stop_loss_price:.2f}\n"
                                f"💹 Profit target: {profit_target:.2f}\n"
                                f"📊 En posición: {in_position}\n"
                                f"📉 Tope semanal: {highest_high:.2f} | 99% del tope: {threshold_price:.2f}\n"
                            )
                        else:
                            report += (
                                f"\n📈 {symbol}\n"
                                f"💰 RSI: {rsi:.2f} | EMA12: {ema:.8f} | Precio actual: {current_price:.8f}\n"
                                f"💰 Compra: {buy_price:.8f} | StopLoss: {stop_loss_price:.8f}\n"
                                f"💹 Profit target: {profit_target:.8f}\n"
                                f"📊 En posición: {in_position}\n"
                                f"📉 Tope semanal: {highest_high:.8f} | 99% del tope: {threshold_price:.8f}\n"
                            )
            
                    except Exception as e:
                        report += f"\n⚠️ Error con {symbol}: {str(e)}\n"
            
                # Balance general
                try:
                    balance_data = binance.fetch_balance()
                    report += f"\n💼 Balance general:\n"
                
                    for coin in ['USDT', 'USDC', 'BONK', 'PEPE', 'HAEDAL', 'EUR', 'SHIB','MEME', 'FDUSD', 'HMSTR', 'BTC']:
                        amount = balance_data['total'].get(coin, 0)
                        if amount > 0:
                            # Determinar contra qué moneda cotizar (quote)
                            if coin in ['BONK', 'HAEDAL', 'BTC']:
                                quote = 'USDC'
                            elif coin in ['PEPE', 'EUR', 'HMSTR', 'SHIB']:
                                quote = 'USDT'
                            elif coin == 'MEME':
                                quote = 'FDUSD'
                            else:
                                quote = None  # Ya son monedas base como USDT o USDC
                
                            if quote:
                                try:
                                    price = binance.fetch_ticker(f'{coin}/{quote}')['last']
                                    estimated = amount * price
                                    if coin == 'BTC':
                                        report += f" - {coin}: {amount:.8f} ≈ {estimated:.4f} {quote}\n"
                                    else:
                                        report += f" - {coin}: {amount:.4f} ≈ {estimated:.4f} {quote}\n"
                                except:
                                    report += f" - {coin}: {amount:.4f} (sin cotización)\n"
                            else:
                                report += f" - {coin}: {amount:.4f}\n"
                
                except Exception as e:
                    report += f"\n⚠️ Error al obtener balances: {str(e)}\n"
            
                send_telegram(report)
                last_report_time = current_time

            time.sleep(1)

        except Exception as e:
            print(f"Error: {str(e)}")
            send_telegram(f"⚠️ Error: {str(e)}")
            time.sleep(60)

if __name__ == "__main__":
    main()
